--[[
    BlockPooky Cooldown Bars Module
    
    This module provides custom cooldown tracking bars for any ability or effect.
    Users can create personalized bars to track:
    - Ability cooldowns (uses configured duration)
    - Effect durations (uses begin/end times from events)
    
    Features:
    - Dynamic creation/removal through settings menu
    - Movable positioning with persistence
    - Customizable colors per bar
    - Character-specific visibility settings
    - Account-wide configuration storage
    
    Bar Types:
    - "ability": Tracks when you cast an ability, shows configured cooldown
    - "effect": Tracks when an effect is gained/lost, uses event timing
    
    CLOSURE PATTERNS EXPLAINED:
    
    This module demonstrates advanced closure usage for dynamic event handling.
    Three distinct closure patterns are employed:
    
    1. FACTORY CLOSURES (OnCooldownAbilityEventClosure, OnCooldownEffectClosure):
       - Create customized event handlers for each cooldown bar
       - Capture the bar 'name' to ensure each handler processes only its own events
       - Essential for dynamic bars where multiple instances need separate handlers
    
    2. IIFE CLOSURES (showCDbar update function):
       - Immediately Invoked Function Expression that captures variables
       - Prevents variable reference issues in timer callbacks
       - Each bar gets its own update loop with correct variable scope
    
    3. SIMPLE CLOSURES (OnMoveStop handlers):
       - Capture 'name' parameter for position saving callbacks
       - Standard pattern for UI event handlers that need context
    
    WHY CLOSURES ARE NECESSARY HERE:
    - ESO's event system requires function references, not method calls
    - Multiple cooldown bars need isolated event processing
    - Dynamic creation means handlers can't be predefined
    - Variable capture prevents reference bugs in async operations
--]]

--[[ basic initialization -------------------------------------------------------------------------------------------]]
BlockPooky = BlockPooky or {}
local BlockPooky = BlockPooky

--[[ cooldown bars implementation ----------------------------------------------------------------------------------]]

---Sets the color of a cooldown bar based on its configuration
---@param name string the name of the cooldown bar to color
function BlockPooky:SetCooldownBarColour(name)
    if self.config.cooldownbar[name].color then
        self.cooldownbar[name].statusBar:SetColor(unpack(self.config.cooldownbar[name].color))
    else
        self.cooldownbar[name].statusBar:SetColor(1, 0, 1, 1) -- Default magenta color
    end
end

function BlockPooky:InitCooldownBarUI(name)
    -- Prüfen, ob die ccBar bereits existiert
    if self.cooldownbar == nil then
        self.cooldownbar = {}
    end
    if self.cooldownbar[name] == nil then
        local config = self:AddCooldownBar(name)
        self.cooldownbar[name] = {}
        self.cooldownbar[name]["bar"] = CreateControl(self.name..name.."Bar", GuiRoot, CT_TOPLEVELCONTROL)
        self.cooldownbar[name]["label"] = CreateControl(self.name..name.."Label", self.cooldownbar[name].bar , CT_LABEL)
        self.cooldownbar[name]["statusBar"] = CreateControl(self.name..name.."Status", self.cooldownbar[name].bar, CT_STATUSBAR)
        
        local cdbar = self.cooldownbar[name]["bar"]

        self.cooldownbar[name].bar:SetDimensions(200, 40)
        self.cooldownbar[name].bar:SetAnchor(CENTER, GuiRoot, CENTER, 0, -120)
        self.cooldownbar[name].bar:SetHidden(true)
        self.cooldownbar[name].bar:SetMovable(true) -- Verschiebbar machen
        self.cooldownbar[name].bar:SetMouseEnabled(true) -- Mausinteraktionen erlauben
        -- SIMPLE CLOSURE: Captures 'name' for the position saving callback
        -- Without this closure, the handler wouldn't know which bar was moved
        self.cooldownbar[name].bar:SetHandler("OnMoveStop", function() self:SaveCooldownBarPosition(name) end)
    
        self.cooldownbar[name].label:SetFont("ZoFontWinH4")
        self.cooldownbar[name].label:SetColor(1, 1, 1, 1) -- Weiß
        self.cooldownbar[name].label:SetText("")
        self.cooldownbar[name].label:SetHorizontalAlignment(TEXT_ALIGN_CENTER)
        self.cooldownbar[name].label:SetAnchor(TOP, cdbar, TOP, 0, 0)

        self.cooldownbar[name].statusBar:SetDimensions(200, 20)
        self.cooldownbar[name].statusBar:SetAnchor(BOTTOM, cdbar, BOTTOM, 0, 0)
        self.cooldownbar[name].statusBar:SetMinMax(0, 1)
        self:SetCooldownBarColour(name)

        if config.left==nil or config.top==nil then
            self:SaveCooldownBarPosition(name)
        end
    end
    -- Lade die gespeicherte Position
    self:LoadCooldownBarPosition(name)
end

function BlockPooky:SaveCooldownBarPosition(name)
    local left, top = self.cooldownbar[name].bar:GetLeft(), self.cooldownbar[name].bar:GetTop()
    if self.config.cooldownbar[name] then
        self.config.cooldownbar[name].left = left
        self.config.cooldownbar[name].top = top
    else
        d("THIS should never happen...")
        self.config.cooldownbar[name] = {
            left = left, top = top, abilityId=0, cooldown=0, type="ability", color={1, 0, 1, 1}}
    end
end

function BlockPooky:LoadCooldownBarPosition(name)
    if self.config and self.config.cooldownbar and self.config.cooldownbar[name] then
        if self.cooldownbar[name] then
            local bar = self.cooldownbar[name].bar
            if bar:GetAnchor() ~= nil then
                bar:ClearAnchors()
            end
            bar:SetAnchor(TOPLEFT, GuiRoot, TOPLEFT, self.config.cooldownbar[name].left, self.config.cooldownbar[name].top)
        end
    end
end

function BlockPooky:GetShowCooldownBar(name)
    if self.toonConfig.cooldownbar[name] then
        return self.toonConfig.cooldownbar[name].show
    end
    return false
end

function BlockPooky:AddCooldownBar(name)
    if self.config.cooldownbar[name] == nil then
        self.config.cooldownbar[name] = {
            top=nil, left=nil, abilityId=0, cooldown=0, type="ability", color={1, 0, 1, 1}}
    end
    if self.toonConfig.cooldownbar[name] == nil then
        self.toonConfig.cooldownbar[name] = {show=false}
    end
    return self.config.cooldownbar[name]
end

function BlockPooky:RemoveCooldownBar(name)
    if self.config.cooldownbar[name] ~= nil then
        self.config.cooldownbar[name] = nil
    end
end

function BlockPooky:InitCooldownBarUIs()
    if self.config.cooldownbar then
        for name, config in pairs(self.config.cooldownbar) do
            self:InitCooldownBarUI(name)
        end
    end
end

function BlockPooky:InitCooldownBarEvents()
    if self.config.cooldownbar then
        for name, config in pairs(self.config.cooldownbar) do
            if self:GetShowCooldownBar(name) then
                self:CooldownBarEventRegisterUpdate(name, true, config.type)
            end
        end
    end
end

function BlockPooky:CooldownBarsSetHidden(hide)
    if self.config.cooldownbar then
        for name, config in pairs(self.config.cooldownbar) do
            if self:GetShowCooldownBar(name) and self.cooldownbar[name] then
                self.cooldownbar[name].bar:SetHidden(hide)
            end
        end
    end
end

function BlockPooky:ResetCooldownBarsPosition()
    if self.config.cooldownbar and self.cooldownbar then
        for name, config in pairs(self.config.cooldownbar) do
            if self:GetShowCooldownBar(name) and self.cooldownbar[name] then
                local bar = self.cooldownbar[name].bar
                if bar:GetAnchor() ~= nil then
                    bar:ClearAnchors()
                end
                bar:SetAnchor(CENTER, GuiRoot, CENTER, 0, -120)
                self:SaveCooldownBarPosition(name)
            end
        end
    end
end

function BlockPooky:RestoreCooldownBarsPosition()
    if self.config.cooldownbar and self.cooldownbar then
        for name, config in pairs(self.config.cooldownbar) do
            if self:GetShowCooldownBar(name) then
                self:LoadCooldownBarPosition(name)
            end
        end
    end
end

function BlockPooky:CooldownBarEventRegisterUpdate(name, useCooldownBar, type)
    if type=="effect" then
        if useCooldownBar then
            EVENT_MANAGER:RegisterForEvent(self.name .. name .."CDWatcher", EVENT_EFFECT_CHANGED, self:OnCooldownEffectClosure(name))
            EVENT_MANAGER:AddFilterForEvent(self.name .. name .."CDWatcher", EVENT_EFFECT_CHANGED, REGISTER_FILTER_UNIT_TAG, "player")
        else
            EVENT_MANAGER:UnregisterForEvent(self.name .. name .."CDWatcher")
        end
    else
        if useCooldownBar then
            EVENT_MANAGER:RegisterForEvent(self.name .. name .."CDWatcher", EVENT_COMBAT_EVENT, self:OnCooldownAbilityEventClosure(name))
            EVENT_MANAGER:AddFilterForEvent(self.name .. name .."CDWatcher", EVENT_COMBAT_EVENT, REGISTER_FILTER_UNIT_TAG, "player")
        else
            EVENT_MANAGER:UnregisterForEvent(self.name .. name .."CDWatcher")
        end
    end
end

function BlockPooky:UpdateCooldownBarUsage(name, useCooldownBar)
    if self.toonConfig.cooldownbar[name] == nil then
        self.toonConfig.cooldownbar[name] {show=false}
    end
    if useCooldownBar ~= self.toonConfig.cooldownbar[name].show then
        if useCooldownBar then
            self:InitCooldownBarUI(name)
            self.toonConfig.cooldownbar[name].show=true
        else
            self.toonConfig.cooldownbar[name].show=false
            if self.cooldownbar and self.cooldownbar[name] then
                self.cooldownbar[name].bar:SetHidden(true)
            end
        end
        self:CooldownBarEventRegisterUpdate(name, useCooldownBar, self.config.cooldownbar[name].type)
    end
end

local BlockPooky_cdBarActiveEndTime = {}
function BlockPooky:showCDbar(name, beginTime, endTime)
    local showed = false;
    local barActiveEndTime = BlockPooky_cdBarActiveEndTime[name]
    if barActiveEndTime == nil then
        barActiveEndTime = 0
    end
    if endTime > barActiveEndTime then
        showed = true;
        BlockPooky_cdBarActiveEndTime[name] = endTime
        local duration = endTime - beginTime
        local cooldownbar = self.cooldownbar[name]
        if cooldownbar.bar ~=nil then
            cooldownbar.statusBar:SetMinMax(0, duration)
            cooldownbar.statusBar:SetValue(duration)
            cooldownbar.label:SetText(name) -- Text über dem Balken
            cooldownbar.bar:SetHidden(false)
        end
        -- Update function for the cooldown bar - CLOSURE PATTERN EXPLANATION:
        -- This is an Immediately Invoked Function Expression (IIFE) closure that creates
        -- a unique update function for each cooldown bar. The pattern works as follows:
        --
        -- 1. (function(name) ... end)(name) - IIFE that captures the current 'name' value
        -- 2. The inner 'function inner()' has access to the captured 'name' variable
        -- 3. This prevents all bars from sharing the same 'name' reference
        -- 4. Each bar gets its own dedicated update function with its specific name
        --
        -- Without this closure pattern, all bars would reference the final 'name' value
        -- from the loop, causing incorrect behavior where all bars update the same UI element.
        EVENT_MANAGER:RegisterForUpdate(self.name..name.."UpdateCDBar", 50, (function(name)
                function inner()
                    local remaining = BlockPooky_cdBarActiveEndTime[name] - GetGameTimeSeconds()
                    local cooldownbar = BlockPooky.cooldownbar[name]
                    if cooldownbar~=nil then
                        if remaining <= 0 then
                            cooldownbar.bar:SetHidden(not locked)
                            EVENT_MANAGER:UnregisterForUpdate(self.name..name.."UpdateCDBar")
                        else
                            cooldownbar.statusBar:SetValue(remaining)
                        end
                    end
                end
                return inner
            end)(name)  -- The 'name' parameter is passed to and captured by the IIFE
        )
    end 
    return showed
end

--[[ event handling - factory closure patterns ---------------------------------------------------------------------]]

--[[
    FACTORY CLOSURE PATTERN - OnCooldownAbilityEventClosure
    
    This is a "factory function" that creates customized event handlers for each cooldown bar.
    The closure pattern solves a critical problem in dynamic event handling:
    
    PROBLEM: Each cooldown bar needs its own EVENT_COMBAT_EVENT handler, but they must
    remember which specific bar (name) they belong to when the event fires.
    
    SOLUTION: This factory function takes a 'name' parameter and returns a specialized
    event handler that has "captured" that specific name in its closure scope.
    
    HOW IT WORKS:
    1. BlockPooky:OnCooldownAbilityEventClosure("MyBar") is called
    2. The 'name' parameter ("MyBar") is captured in the closure
    3. An inner function OnCooldownAbilityEvent is created with access to that 'name'
    4. The inner function is returned and used as the actual event handler
    5. When events fire, the handler knows it belongs to "MyBar" specifically
    
    BENEFITS:
    - Each cooldown bar gets its own dedicated event handler
    - No global state or lookup tables needed
    - Clean separation - each handler only processes its own bar's events
    - Memory efficient - handlers are created only when bars are configured
    
    USAGE EXAMPLE:
    local myBarHandler = self:OnCooldownAbilityEventClosure("MyBar")
    EVENT_MANAGER:RegisterForEvent("MyBarWatcher", EVENT_COMBAT_EVENT, myBarHandler)
--]]
---Creates a specialized EVENT_COMBAT_EVENT handler for a specific cooldown bar
---@param name string the name of the cooldown bar this handler belongs to
---@return function event handler function that captures the bar name
function BlockPooky:OnCooldownAbilityEventClosure(name)
    function OnCooldownAbilityEvent(
        eventCode, result, isError, abilityName, abilityGraphic, abilityActionSlotType,
        sourceName, sourceType, targetName, targetType, hitValue, powerType, damageType,
        combat_log, sourceUnitId, targetUnitId, abilityId)
        local cleanSourceName = self:CleanupName(sourceName)
        -- local cleanAbilityName = self:CleanupName(abilityName)
        if cleanSourceName == self.player or cleanSourceName ~= self.companionName then
            -- Access the specific cooldown bar config using the captured 'name'
            local config = self.config.cooldownbar[name]
            if config~=nil then
                if abilityId == tonumber(config.abilityId) then
                    if result == ACTION_RESULT_EFFECT_GAINED then
                        beginTime = GetGameTimeSeconds()
                        endTime = beginTime + config.cooldown
                        self:showCDbar(name, beginTime, endTime)
                    elseif result == ACTION_RESULT_EFFECT_FADED then
                        self.cooldownbar[name].bar:SetHidden(not locked)
                        EVENT_MANAGER:UnregisterForUpdate(self.name..name.."UpdateCDBar")
                    end
                end
            end
        end
    end
    return OnCooldownAbilityEvent  -- Return the closure with captured 'name'
end

--[[
    FACTORY CLOSURE PATTERN - OnCooldownEffectClosure
    
    This is the companion factory function for EVENT_EFFECT_CHANGED handlers.
    It follows the same closure pattern as OnCooldownAbilityEventClosure but handles
    effect-based cooldown bars instead of ability-based ones.
    
    KEY DIFFERENCES FROM ABILITY HANDLER:
    - Listens to EVENT_EFFECT_CHANGED instead of EVENT_COMBAT_EVENT
    - Uses effect begin/end times when available
    - Falls back to configured cooldown if begin == end (some effects don't provide duration)
    - Handles EFFECT_RESULT_GAINED/FADED instead of ACTION_RESULT_EFFECT_GAINED/FADED
    
    SAME CLOSURE BENEFITS:
    - Each effect-based cooldown bar gets its own dedicated handler
    - The 'name' parameter is captured and available when events fire
    - Clean separation between different bars
    - No cross-contamination between different effect trackers
--]]
---Creates a specialized EVENT_EFFECT_CHANGED handler for a specific cooldown bar
---@param name string the name of the cooldown bar this handler belongs to
---@return function event handler function that captures the bar name
function BlockPooky:OnCooldownEffectClosure(name)
    function OnCooldownEffectEvent(
        eventCode, changeType, effectSlot, effectName, unitTag, beginTime, endTime, stackCount,
        iconName, buffType, effectType, abilityType, statusEffectType, unitName, unitId, abilityId, sourceType)
        if unitTag ~= "player" then return end -- Only relevant for the player
        -- Access the specific cooldown bar config using the captured 'name'
        local config = self.config.cooldownbar[name]
        if config~=nil then
            if abilityId == tonumber(config.abilityId) then
                if changeType == EFFECT_RESULT_GAINED then
                    -- Some effects don't provide duration (beginTime == endTime)
                    -- In those cases, use the configured cooldown duration
                    if beginTime == endTime then
                        endTime = beginTime + config.cooldown
                    end
                    self:showCDbar(name, beginTime, endTime)
                elseif changeType == EFFECT_RESULT_FADED then
                    self.cooldownbar[name].bar:SetHidden(not locked)
                    EVENT_MANAGER:UnregisterForUpdate(self.name..name.."UpdateCDBar")
                end
            end
        end
    end
    return OnCooldownEffectEvent  -- Return the closure with captured 'name'
end  