--[[ basic initialization -------------------------------------------------------------------------------------------]]
BlockPooky = BlockPooky or {}
local BlockPooky = BlockPooky

local BlockPooky_threatAlertActive = false
local BlockPooky_threatAlertTimer = nil
local BlockPooky_lastThreatAlert = 0

-- Threat Ability IDs for Dark Convergence, Rush of Agony, and other dangerous abilities
BlockPooky.THREAT_ABILITY_IDS = {
    159385, -- Dark Convergence (ability ID)
    159279, -- Rush of Agony (ability ID)
}
BlockPooky.THREAT_ABILITY_NAMES = nil


--[[ Threat Alert implementation --------------------------------------------------------------------------------------]]

function BlockPooky:SetThreatAlertColor()
    if BlockPookyThreatAlertTexture then
        -- Set dimensions to screen size
        local screenWidth, screenHeight = GuiRoot:GetDimensions()
        BlockPookyThreatAlert:SetDimensions(screenWidth, screenHeight)
        BlockPookyThreatAlertTexture:SetDimensions(screenWidth, screenHeight)
        
        -- Set texture from config
        local textureFile = self.config.threatalert.texture or "red.dds"
        local texturePath = "BlockPooky/textures/" .. textureFile
        BlockPookyThreatAlertTexture:SetTexture(texturePath)
        
        -- Ensure overlay starts hidden
        BlockPookyThreatAlert:SetAlpha(0.0)
        BlockPookyThreatAlertTexture:SetAlpha(0.0)
    else
        d("ERROR: BlockPookyThreatAlertTexture not found!")
    end
end

function BlockPooky:InitThreatAlert()
    if not self.THREAT_ABILITY_NAMES then
        self.THREAT_ABILITY_NAMES = {}
        for idx = #self.THREAT_ABILITY_IDS, 1, -1 do
            local cleanName = self:CleanAbilityName(self.THREAT_ABILITY_IDS[idx])
            self.THREAT_ABILITY_NAMES[cleanName] = self.THREAT_ABILITY_IDS[idx]
        end
    end
    self:RegisterThreatAlert()
end

function BlockPooky:RegisterThreatAlert()
    if self.config.threatalert and self.config.threatalert.show then
        EVENT_MANAGER:RegisterForEvent(self.name .. "ThreatAlert", EVENT_COMBAT_EVENT, function(...) self:OnThreatAlert(...) end)
        EVENT_MANAGER:AddFilterForEvent(self.name .. "ThreatAlert", EVENT_COMBAT_EVENT, REGISTER_FILTER_COMBAT_RESULT, ACTION_RESULT_EFFECT_GAINED)
        EVENT_MANAGER:AddFilterForEvent(self.name .. "ThreatAlert", EVENT_COMBAT_EVENT, REGISTER_FILTER_IS_ERROR, false)
    end
end

function BlockPooky:UnRegisterThreatAlert()
    EVENT_MANAGER:UnregisterForEvent(self.name .. "ThreatAlert")
end

function BlockPooky:ShowThreatAlert(duration)
    if BlockPookyThreatAlert then
        local alpha = self.config.threatalert.alpha or 0.3
        BlockPookyThreatAlert:SetAlpha(alpha)
        BlockPookyThreatAlertTexture:SetAlpha(alpha)
        BlockPooky_threatAlertActive = true
        
        -- Clear any existing timer
        if BlockPooky_threatAlertTimer then
            zo_removeCallLater(BlockPooky_threatAlertTimer)
        end
        
        -- Use actual effect duration (in milliseconds from event) or default to 8 seconds
        local hideDuration = duration or 8000
        BlockPooky_threatAlertTimer = zo_callLater(function() self:HideThreatAlert() end, hideDuration)
    end
end

function BlockPooky:HideThreatAlert()
    if BlockPookyThreatAlert then
        BlockPookyThreatAlert:SetAlpha(0.0)
        BlockPookyThreatAlertTexture:SetAlpha(0.0)
        BlockPooky_threatAlertActive = false
        
        -- Clear timer if active
        if BlockPooky_threatAlertTimer then
            zo_removeCallLater(BlockPooky_threatAlertTimer)
            BlockPooky_threatAlertTimer = nil
        end
    end
end


--[[ event handling -------------------------------------------------------------------------------------------------]]

function BlockPooky:OnThreatAlert(eventCode, result, isError, abilityName, abilityGraphic, abilityActionSlotType, sourceName, sourceType, targetName, targetType, hitValue, powerType, damageType, log, sourceUnitId, targetUnitId, abilityId, overflow)
    -- Skip if disabled
    if not self.config.threatalert.show then
        return
    end
    
    -- Check if PvP only and not in Cyrodiil
    if self.config.threatalert.pvpOnly and not self:IsInCyro() then
        return
    end
    
    -- Only trigger on ability effects cast by enemies (not player, not group members)
    if sourceType == COMBAT_UNIT_TYPE_PLAYER or isError then
        return
    end
    
    -- Check if source is player or group member (same checks as block warning)
    if sourceName == self.player or sourceName == self.companionName or self.group[sourceName] then
        return
    end
    
    local function isThreatAbility(element)
        return BlockPooky.THREAT_ABILITY_NAMES[element] ~= nil
    end
    
    local cleanName = self:CleanupName(abilityName)
    
    if isThreatAbility(cleanName) and not BlockPooky_threatAlertActive then
        -- Apply cooldown to prevent spam (same as block warning)
        local now = GetGameTimeMilliseconds()
        if (now - BlockPooky_lastThreatAlert) < (self.config.threatalert.cooldown or 5000) then
            return
        end
        BlockPooky_lastThreatAlert = now
        
        -- Get duration from config or use default
        local duration = self.config.threatalert.duration or 8000 -- 8 seconds default
        self:ShowThreatAlert(duration)
    end
end
